C**********************************************************************
C
C     SUBROUTINE NAME: DLIBPROP
C
C     CREATED ON: 2003-04-14     BY: Brian C. Watson
C
C     PURPOSE: EXAMPLE ROUTINE FOR USER CROSS SECTION ELEMENTS.
C              INTERFACE 1 -- CALCULATE ELEMENT PROPERTIES FROM
C                             CROSS-SECTION DIMENSIONS
C
C     PARAMETER LIST: (CSD,V,SP,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: IWHICH - WHICH USER ELEMENT TYPE TO EVALUATE (15-25).
C            CSD(*) - THE ELEMENT CROSS SECTIONAL DIMESIONS.
C            V      - POISSON RATIO.
C
C     OUTPUT: SP(*) - THE ELEMENT SECTION PROPERTIES.
C            IERR   - THE ERROR SWITCH.
C                    =0 FOR NO ERROR.
C                    =1 FOR ERROR.
C                    =2 FOR NO USER SUPPLIED ROUTINE.
C
C**********************************************************************
      SUBROUTINE DLIBPROP(IWHICH,CSD,V,SP,IERR)
CDEC$ ATTRIBUTES DLLEXPORT :: DLIBPROP
      IMPLICIT NONE
C
      INTEGER IWHICH, IERR
C
      DOUBLE PRECISION CSD(*), V
C
      DOUBLE PRECISION SP(*)
C
      IF (IWHICH.EQ.22) THEN
         CALL MYPROP22(CSD, V, SP, IERR)
      ELSE IF (IWHICH.EQ.23) THEN
         CALL MYPROP23(CSD, V, SP, IERR)
      ELSE
         IERR = 2
      ENDIF

      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: DLIBSTRESS
C
C     CREATED ON: 2003-04-14     BY: Brian C. Watson
C
C     PURPOSE: EXAMPLE ROUTINE FOR USER CROSS SECTION ELEMENTS.
C              INTERFACE 2 -- CALCULATE ELEMENT STRESSES FROM
C                             CROSS-SECTION DIMENSIONS AND FORCES
C
C     PARAMETER LIST: (CSD,V,SP,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: IWHICH - WHICH USER ELEMENT TYPE TO EVALUATE (15-25).
C            CSD(*) - THE ELEMENT CROSS SECTIONAL DIMESIONS.
C            FORCE(*) - ELEMENT FORCES
C
C     OUTPUT: STRESS(*) - THE ELEMENT STRESSES.
C            IERR   - THE ERROR SWITCH.
C                    =0 FOR NO ERROR.
C                    =1 FOR ERROR.
C                    =2 FOR NO USER SUPPLIED ROUTINE.
C
C**********************************************************************
      SUBROUTINE DLIBSTRESS(IWHICH,CSD,FORCE,STRESS,IERR)
CDEC$ ATTRIBUTES DLLEXPORT :: DLIBSTRESS
      IMPLICIT NONE
C
      INTEGER IWHICH, IERR
C
      DOUBLE PRECISION CSD(*), FORCE(*)
C
      DOUBLE PRECISION STRESS(*)
C
      IF (IWHICH.EQ.22) THEN
         CALL MYSTRESS22(CSD,FORCE,STRESS,IERR)
      ELSE IF (IWHICH.EQ.23) THEN
         CALL MYSTRESS23(CSD,FORCE,STRESS,IERR)
      ELSE
         IERR = 2
      ENDIF

      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: GNLB22
C
C     CREATED ON:      8/31/92        BY: HAROLD THOMAS
C
C     PURPOSE: CHECKING ROUTINE FOR DESIGN LIBRARY PLATE ELEMENTS.
C
C     PARAMETER LIST: (CSD,V,SP,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: CSD(2) - THE ELEMENT CROSS SECTIONAL DIMESIONS.
C            V        - POISSON RATIO.
C            IERR   - THE ERROR SWITCH.
C
C     OUTPUT: SP(2) - THE ELEMENT SECTION PROPERTIES.
C
C     CALLED BY: GN61UP, GN16UP,GN40LB
C
C     CALLS: NONE
C
C     ALGORITHM:
C
C     1. CALCULATE SECTION PROPERTIES.
C
C**********************************************************************
C
      SUBROUTINE MYPROP22(CSD,V,SP,IERR)
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DOUBLE PRECISION CSD(1), SP(2),V
C
      IERR = 0
C
C     SOLID PLATE
C
      T=CSD(1)
C
      SP(1) = T
      SP(2) = T**3/12.0D0
C
      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: MYSTRESS22
C
C     CREATED ON: 8/31/92            BY: HAROLD THOMAS
C
C     PURPOSE: CHECKING ROUTINE FOR PLATE DESIGN ELEMENT.
C
C     PARAMETER LIST: (X,F,S,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: X(1) - THE ELEMENT CROSS SECTIONAL DIMENSIONS.
C            F(8) - THE ELEMENT FORCES.
C
C     OUTPUT: S(14) - THE ELEMENT STRESSES.
C             IERR  - THE ERROR SWITCH.
C
C     CALLED BY: 
C
C     CALLS: PRINC2
C
C     ALGORITHM:
C
C     1. CALCULATE ELEMENT STRESSES.
C
C**********************************************************************
C
      SUBROUTINE MYSTRESS22(X,F,S,IERR)
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DOUBLE PRECISION X(1), F(8), S(14) 
C
      PARAMETER (TWO=2.0D0,TWELVE=12.0D0)
C
      IERR = 0
C
C     SOLID PLATE
C
      T=X(1)
      T3 = T*T*T
      Z1 = -T/TWO
      Z2 = T/TWO
      ZZ1 =  TWELVE*Z1/T3
      ZZ2 =  TWELVE*Z2/T3
C
      S(5) = F(1)/T - F(6)*ZZ1
      S(6) = F(2)/T - F(7)*ZZ1
      S(7) = F(3)/T - F(8)*ZZ1
      S(12) = F(1)/T - F(6)*ZZ2
      S(13) = F(2)/T - F(7)*ZZ2
      S(14) = F(3)/T - F(8)*ZZ2
      CALL PRINC2(S(5),S(1),THETA,1,0.0D0)
      CALL PRINC2(S(12),S(8),THETA,1,0.0D0)
C
      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: PRINC2
C
C     CREATED ON: 8/23/90         BY: HAROLD THOMAS
C
C     PURPOSE: CALCULATE THE PRINCIPLE STRESSES AND STRIANS IN A 2-D
C              PROBLEM
C
C     PARAMETER LIST: (SL,SP,THETA,ISN,SIGZ)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: SL(3) - THE LOCAL STRESS VECTOR.
C            ISN   - THE STRESS (ISN=1) OR STRAIN (ISN=2) SWITCH.
C            SIGZ  - THE SIGMA-Z STRESS (USED FOR PLANE STRAIN 
C                    PROBLEMS).
C
C     OUTPUT: SP(4) - THE PRINCIPLE STRESS VECTOR.
C             THETA - THE ANGLE OF THE PRINCIPLE STRESSES.
C
C     CALLED BY: GN40S4
C
C     CALLS: NONE
C
C     ALGORITHM:
C
C     1. DO IT TO IT.
C
C********************************************************************
C
      SUBROUTINE PRINC2(SL,SP,THETA,ISN,SIGZ)
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C    
      PARAMETER (EPS=1.0D-7, EPS1=1.0D-30)
C
      DOUBLE PRECISION SL(3), SP(4)
C
      SX =  SL(1)
      SY =  SL(2)
      SXY = SL(3)
      IF (ISN.EQ.2) SXY = SXY/2.0D0
C
      C1 = (SX + SY)/2.0D0
C
      IF (ABS(SX-SY).LE.EPS*ABS(SXY)) THEN
         C2 = ABS(SXY)
      ELSE
         C2 = SQRT(((SX - SY)/2.0D0)**2 + SXY**2)
         IF (C2.LT.ABS(SXY)) C2 = ABS(SXY)
      END IF
C
      SP(3) = C1 + C2
      SP(4) = C1 - C2
C
      IF (C2.GT.EPS1) THEN
         SXYC2 = SXY/C2
         IF (SXYC2.GE. 1.0D0) SXYC2 =  1.0D0
         IF (SXYC2.LE.-1.0D0) SXYC2 = -1.0D0
         THETA = (ASIN(SXYC2))/2.0D0 
         THETA = SIGN(THETA,SXY*(SX-SY))
      ELSE
         THETA = 0.0D0
      END IF
C
      THETA = THETA*180.0D0/3.1415926540D0
C
      IF (ISN.EQ.1) THEN
C
         SP(1) = C2
         SP(2) = SQRT(((SX-SY)**2 + (SY-SIGZ)**2 + (SIGZ - SX)**2)/2.0D0
     *                + 3.0D0*SXY**2)
C
      ELSE
C
         SXY = SXY*2.0D0
         SP(1) = SQRT((SX - SY)**2 + SXY**2)
         SP(2) = SQRT(4.0D0*(SX**2+SY**2 - SX*SY)/9.0D0 + SXY**2/3.0D0)
C
      END IF
C
      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: MYPROP23
C
C     CREATED ON:      8/31/92        BY: HAROLD THOMAS
C
C     PURPOSE: CHECKING ROUTINE FOR DESIGN LIBRARY BEAM ELEMENTS.
C
C     PARAMETER LIST: (CSD,V,SP,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: CSD(2) - THE ELEMENT CROSS SECTIONAL DIMESIONS.
C            V        - POISSON RATIO.
C            IERR   - THE ERROR SWITCH.
C
C     OUTPUT: SP(2) - THE ELEMENT SECTION PROPERTIES.
C
C     CALLED BY: GN61UP, GN16UP,GN40LB
C
C     CALLS: NONE
C
C     ALGORITHM:
C
C     1. CALCULATE SECTION PROPERTIES.
C
C**********************************************************************
C
      SUBROUTINE MYPROP23(CSD,V,SP,IERR)
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DOUBLE PRECISION CSD(2), SP(2),V
C
      IERR = 0
C
      B=CSD(1)
      H=CSD(2)
      A = B*H
      RI = B*H*H*H/12.0D0
C
      SP(1) = A
      SP(2) = RI
C
      RETURN
      END
C**********************************************************************
C
C     SUBROUTINE NAME: MYSTRESS23
C
C     CREATED ON: 8/31/92            BY: HAROLD THOMAS
C
C     PURPOSE: CHECKING ROUTINE FOR BEAM DESIGN ELEMENT.
C
C     PARAMETER LIST: (X,F,S,IERR)
C
C     COMMON BLOCKS USED: NONE
C
C     INPUT: X(2) - THE ELEMENT CROSS SECTIONAL DIMENSIONS.
C            F(8) - THE ELEMENT FORCES.
C
C     OUTPUT: S(2) - THE ELEMENT STRESSES.
C             IERR  - THE ERROR SWITCH.
C
C     CALLED BY: GN40BS, GN40S4
C
C     CALLS: GN40P2
C
C     ALGORITHM:
C
C     1. CALCULATE ELEMENT STRESSES.
C
C**********************************************************************
C
      SUBROUTINE MYSTRESS23(X,F,S,IERR)
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DOUBLE PRECISION X(2), F(12), S(2) 
C
      PARAMETER (TWO=2.0D0,TWELVE=12.0D0)
C
      IERR = 0
C
C     SOLID PLATE
C
      B=X(1)
      H=X(2)
      A = B*H
      RI = B*H*H*H/TWELVE
      C1 = B/TWO
      C2 = B/TWO
      RM1 = F(5)
      RM2 = F(12)
C
      S(1) = F(1)/A - RM1*C1/RI
      S(2) = F(1)/A - RM2*C2/RI
C
      RETURN
      END
